/// @file dltLogger.cpp
///
/// @brief Implements the DLT specific logger
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see Log SyslogLogger

#include <cstring>
#include <iostream>
#include <string>
#include <dlt/dlt.h>

#include "dltLogger.h"

namespace adit {
namespace uspi {

/// @brief Creates a DltLog object that will register ident as DLT context
/// @param ident The DLT context ID.
///
/// The DLT application must already be registered.
DltLog::DltLog(const char *ident)
{
    mPriority = kLogInfo;
    mDefaultPriority = kLogInfo;

    mIdent.assign(ident);

    dlt_register_context(&dd_dlt,
                         mIdent.c_str(),
                         "Logger context");
}

/// @brief Unregister the DLT context
DltLog::~DltLog()
{
    dlt_unregister_context(&dd_dlt);
}

/// @brief Flushes the @ref mBuffer to DLT if non-empty
///
/// The @ref mPriority is used as log level to be sent with the buffer. Once the
/// buffer is sent, it is erased, and the @ref mPriority is reset to the @ref
/// mDefaultPriority value.
int DltLog::sync()
{
    if (mBuffer.length()) {
        DltContextData log;
        DltLogLevelType prio = toLogPriority(mPriority);

        /* Log the exception message to the viewer */
        if (dlt_user_log_write_start(&dd_dlt,
                                     &log,
                                     prio))
        {
            dlt_user_log_write_string(&log, mBuffer.c_str());
            dlt_user_log_write_finish(&log);
        }

        mBuffer.erase();
        mPriority = mDefaultPriority;
    }

    return 0;
}

/// @brief Add a character to the buffer and sync if EOF is met.
/// @param c The character to push into the @ref mBuffer
int DltLog::overflow(int c)
{
    if (c != EOF)
        mBuffer += static_cast<char>(c);
    else
        sync();

    return c;
}

/// @brief Converts a @ref LogPriority& to a @c DltLogLevelType
/// @param log_priority The priority to convert
DltLogLevelType DltLog::toLogPriority(const LogPriority& log_priority)
{
    switch(log_priority) {
    case kLogEmerg:
    case kLogAlert:
    case kLogCrit:
        return DLT_LOG_FATAL;
    case kLogErr:
        return DLT_LOG_ERROR;
    case kLogWarning:
        return DLT_LOG_WARN;
    case kLogNotice:
    case kLogInfo:
        return DLT_LOG_INFO;
    default:
        return DLT_LOG_DEBUG;
    }
}

} // namespace uspi
} // namespace adit
